<?php
/**
 * User Notification
 *
 * @package    wp-freeio
 * @author     Habq 
 * @license    GNU General Public License, version 3
 */

if ( ! defined( 'ABSPATH' ) ) {
  	exit;
}

class WP_Freeio_User_Notification {
	
	public static function init() {

		add_action( 'wpfi_ajax_wp_freeio_ajax_remove_notify',  array(__CLASS__, 'process_remove_notification') );

		// for private message
		add_action('wp-private-message-after-add-message', array( __CLASS__, 'add_private_message_notify'), 10, 3 );
	}

	public static function add_notification($args) {
		
		$args = wp_parse_args( $args, array(
			'post_type' => 'employer',
			'user_post_id' => 0,
			'unique_id' => uniqid(),
			'viewed' => 0,
            'time' => current_time('timestamp'),
            'type' => '',
            'application_id' => 0,
            'employer_id' => 0,
            'job_id' => 0,
		));

		extract( $args );
		
		if ( empty($user_post_id) || empty($post_type) ) {
			return;
		}

		$prefix = WP_FREEIO_FREELANCER_PREFIX;
		if ( !empty($post_type) && $post_type == 'employer' ) {
			$prefix = WP_FREEIO_EMPLOYER_PREFIX;
		}
		$notifications = get_post_meta($user_post_id, $prefix . 'notifications', true);
        $notifications = !empty($notifications) ? $notifications : array();

        $new_notifications = array_merge( array($unique_id => $args), $notifications );

		update_post_meta($user_post_id, $prefix . 'notifications', $new_notifications);
	}

	public static function process_remove_notification() {
		$return = array();
		if (  !isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'wp-freeio-remove-notify-nonce' )  ) {
			$return = array( 'status' => false, 'msg' => esc_html__('La vérification du nonce a échoué.', 'wp-freeio') );
		   	echo wp_json_encode($return);
		   	exit;
		}

		$unique_id = !empty($_POST['unique_id']) ? $_POST['unique_id'] : '';

		$user_id = WP_Freeio_User::get_user_id();
		if ( WP_Freeio_User::is_employer() ) {
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
			$post_type = 'employer';
			$prefix = WP_FREEIO_EMPLOYER_PREFIX;
		} elseif ( WP_Freeio_User::is_freelancer() ) {
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
			$post_type = 'freelancer';
			$prefix = WP_FREEIO_FREELANCER_PREFIX;
		} else {
			$return = array( 'status' => false, 'msg' => esc_html__('Vous ne pouvez pas supprimer la notification', 'wp-freeio') );
		   	echo wp_json_encode($return);
		   	exit;
		}


		$notifications = self::get_notifications($user_post_id, $post_type);
		if ( !empty($notifications[$unique_id]) ) {
			unset($notifications[$unique_id]);
			update_post_meta($user_post_id, $prefix . 'notifications', $notifications);

			$return = array( 'status' => true, 'msg' => esc_html__('La notification a été supprimée avec succès.', 'wp-freeio') );
		   	echo wp_json_encode($return);
		   	exit;
		} else {
			$return = array( 'status' => false, 'msg' => esc_html__('La notification n\'existe pas.', 'wp-freeio') );
		   	echo wp_json_encode($return);
		   	exit;
		}
		
	}

	public static function get_notifications($user_post_id, $post_type = 'employer') {
		
		if ( empty($user_post_id) || empty($post_type) ) {
			return;
		}

		$prefix = WP_FREEIO_FREELANCER_PREFIX;
		if ( !empty($post_type) && $post_type == 'employer' ) {
			$prefix = WP_FREEIO_EMPLOYER_PREFIX;
		}
		$notifications = get_post_meta($user_post_id, $prefix . 'notifications', true);;
        $notifications = !empty($notifications) ? $notifications : array();

        return $notifications;
	}

	public static function get_not_seen_notifications($user_post_id, $post_type = 'employer') {
		$notifications = self::get_notifications($user_post_id, $post_type);
		if ( empty($notifications) ) {
			return;
		}
		$return = [];
		foreach ( $notifications as $key => $notify ) {
			if ( isset($notify['viewed']) ) {
				$return[] = $notify;
			}
		}
        return $return;
	}
	
	public static function remove_notification($user_post_id, $post_type = 'employer', $unique_id = '') {
		$notifications = self::get_notifications($user_post_id, $post_type);
		if ( empty($notifications) ) {
			return true;
		}
		if ( !empty($notifications[$unique_id]) ) {
			unset($notifications[$unique_id]);
		} else {
			return false;
		}

		$prefix = WP_FREEIO_FREELANCER_PREFIX;
		if ( !empty($post_type) && $post_type == 'employer' ) {
			$prefix = WP_FREEIO_EMPLOYER_PREFIX;
		}

		update_post_meta($user_post_id, $prefix . 'notifications', $notifications);

        return true;
	}
	
	public static function add_private_message_notify($message_id, $recipient, $user_id) {
		if ( WP_Freeio_User::is_employer($recipient) ) {
			$post_type = 'employer';
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($recipient);
		} elseif ( WP_Freeio_User::is_freelancer($recipient) ) {
			$post_type = 'freelancer';
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($recipient);
		}
		
		if ( $post_type == 'employer' || $post_type == 'freelancer' ) {
			$notify_args = array(
				'post_type' => $post_type,
				'user_post_id' => $user_post_id,
	            'type' => 'new_private_message',
	            'user_id' => $user_id,
	            'message_id' => $message_id,
			);
			WP_Freeio_User_Notification::add_notification($notify_args);
		}
	}

	public static function display_notify($args) {
		$type = !empty($args['type']) ? $args['type'] : '';
		switch ($type) {
			case 'email_apply':
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';
				$html = sprintf(__('Une nouvelle candidature a été soumise pour votre offre <a href="%s">%s</a>', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id));
				break;
			case 'internal_apply':
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';
				$freelancer_id = !empty($args['freelancer_id']) ? $args['freelancer_id'] : '';
				$html = sprintf(__('Une nouvelle candidature a été soumise pour votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($freelancer_id), get_the_title($freelancer_id) );
				break;
			case 'remove_apply':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';

				$html = sprintf(__('La candidature a été supprimée de votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'create_meeting':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$post_id = !empty($args['post_id']) ? $args['post_id'] : '';

				$topic_title = '';
				$job_id = 0;
				if ( get_post_type($post_id) == 'job_applicant') {
					$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
					$topic_title = esc_html__('offre', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'project_proposal' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id');
					$topic_title = esc_html__('projet', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'service_order' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id');
					$topic_title = esc_html__('service', 'wp-freeio');
				} else {
					$html = '';
					break;
				}

				$html = sprintf(__('Une nouvelle réunion a été créée sur le %s <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $topic_title, get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'reschedule_meeting':
				$reschedule_user_id = !empty($args['reschedule_user_id']) ? $args['reschedule_user_id'] : '';
				$meeting_id = !empty($args['meeting_id']) ? $args['meeting_id'] : '';
				$post_id = WP_Freeio_Meeting::get_post_meta($meeting_id, 'post_id');

				if ( get_post_type($post_id) == 'job_applicant') {
					$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
					$topic_title = esc_html__('offre', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'project_proposal' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id');
					$topic_title = esc_html__('projet', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'service_order' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id');
					$topic_title = esc_html__('service', 'wp-freeio');
				}

				$html = sprintf(__('Une réunion a été reprogrammée pour le %s <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $topic_title, get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'remove_meeting':
				$post_id = !empty($args['post_id']) ? $args['post_id'] : '';
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$topic_title = $job_id = '';
				if ( get_post_type($post_id) == 'job_applicant') {
					$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
					$topic_title = esc_html__('offre', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'project_proposal' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id');
					$topic_title = esc_html__('projet', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'service_order' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id');
					$topic_title = esc_html__('service', 'wp-freeio');
				}
				$html = '';
				if ( $job_id && $topic_title) {
					$html = sprintf(__('Une réunion a été supprimée sur le %s <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $topic_title, get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				}
				break;
			case 'cancel_meeting':
				$post_id = !empty($args['post_id']) ? $args['post_id'] : '';
				$freelancer_id = !empty($args['freelancer_id']) ? $args['freelancer_id'] : '';
				if ( get_post_type($post_id) == 'job_applicant') {
					$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
					$topic_title = esc_html__('offre', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'project_proposal' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id');
					$topic_title = esc_html__('projet', 'wp-freeio');
				} elseif( get_post_type($post_id) == 'service_order' ) {
					$job_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id');
					$topic_title = esc_html__('service', 'wp-freeio');
				}

				$html = sprintf(__('Une réunion a été annulée sur votre %s <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $topic_title, get_permalink($job_id), get_the_title($job_id), get_permalink($freelancer_id), get_the_title($freelancer_id) );
				break;
			case 'reject_applied':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';

				$html = sprintf(__('La candidature a été refusée pour votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'undo_reject_applied':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';

				$html = sprintf(__('Le rejet de la candidature a été annulé pour votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'approve_applied':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';

				$html = sprintf(__('La candidature a été acceptée pour votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'undo_approve_applied':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$job_id = !empty($args['job_id']) ? $args['job_id'] : '';

				$html = sprintf(__('L\'acceptation de la candidature a été annulée pour votre offre <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($job_id), get_the_title($job_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'new_private_message':
                $user_id = !empty($args['user_id']) ? $args['user_id'] : '';
            
                $messages_url = 'https://www.holistic-babe.fr/messages/';
            
                if ( WP_Freeio_User::is_employer() ) {
                    $user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
                } elseif ( WP_Freeio_User::is_freelancer() ) {
                    $user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
                }
            
                if ( !empty($user_post_id) ) {
                    $html = sprintf(
                        __('Nouveau message privé de <a href="%s">%s</a>. <a href="%s">Voir la conversation</a>', 'wp-freeio'),
                        get_permalink($user_post_id),
                        get_the_title($user_post_id),
                        esc_url($messages_url)
                    );
                } else {
                    $user = get_userdata( $user_id );
                    $html = sprintf(
                        __('Nouveau message privé de %s. <a href="%s">Voir la conversation</a>', 'wp-freeio'),
                        $user->display_name,
                        esc_url($messages_url)
                    );
                }
                break;
			case 'invite_freelancer_apply':
				$project_ids = !empty($args['project_ids']) ? $args['project_ids'] : '';
				
				if ( !empty($project_ids) && count($project_ids) == 1 ) {
					$html = sprintf(__('Vous êtes invité à envoyer une proposition pour ce projet <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($project_ids[0]), get_the_title($project_ids[0]) );
				} elseif( !empty($project_ids) ) {
					$jobs_html = '';
					$count = 1;
					foreach ($project_ids as $project_id) {
						$jobs_html .= '<a href="'.get_permalink($project_id).'">'.get_the_title($project_id).'</a>'.($count < count($project_ids) ? ', ' : '');
						$count++;
					}
					$html = sprintf(__('Vous êtes invité à postuler aux offres %s.', 'wp-freeio'), $jobs_html );
				}
				break;
			case 'new_proposal':
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$freelancer_user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_user_id);
				
				$html = sprintf(__('Une nouvelle proposition a été soumise pour votre projet <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($project_id), get_the_title($project_id), get_permalink($freelancer_id), get_the_title($freelancer_id) );
				break;
			case 'edit_proposal':
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$freelancer_user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_user_id);
				
				$html = sprintf(__('Une proposition a été modifiée pour votre projet <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($project_id), get_the_title($project_id), get_permalink($freelancer_id), get_the_title($freelancer_id) );
				break;
			case 'hired_proposal':
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$employer_user_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_user_id);

				$html = sprintf(__('Vous avez été embauché pour le projet suivant <a href="%s">%s</a> par le client <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($project_id), get_the_title($project_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'cancelled_hired_proposal':
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$employer_user_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_user_id);

				$html = sprintf(__('L\'embauche a été annulée pour le projet suivant <a href="%s">%s</a> par le client <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($project_id), get_the_title($project_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'proposal_message':
				$post_type = !empty($args['post_type']) ? $args['post_type'] : '';
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$proposal_id = !empty($args['proposal_id']) ? $args['proposal_id'] : '';
				$user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				
				$view_history_url = '';
				if ( $post_type == 'freelancer' ) {
					$my_projects_page_id = wp_freeio_get_option('my_projects_page_id');
					$my_projects_url = get_permalink( $my_projects_page_id );

					$my_projects_url = add_query_arg( 'project_id', $project_id, remove_query_arg( 'project_id', $my_projects_url ) );
					$my_projects_url = add_query_arg( 'proposal_id', $proposal_id, remove_query_arg( 'proposal_id', $my_projects_url ) );
					$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_projects_url ) );

					$user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';

					$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
				} elseif ( $post_type == 'employer' ) {
					$my_proposals_page_id = wp_freeio_get_option('my_proposals_page_id');
					$my_proposals_url = get_permalink( $my_proposals_page_id );

					$my_proposals_url = add_query_arg( 'project_id', $project_id, remove_query_arg( 'project_id', $my_proposals_url ) );
					$my_proposals_url = add_query_arg( 'proposal_id', $proposal_id, remove_query_arg( 'proposal_id', $my_proposals_url ) );
					$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_proposals_url ) );

					$user_id = !empty($args['user_post_id']) ? $args['user_post_id'] : '';
					$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
				}

				$html = sprintf(__('Un message a été envoyé sur votre proposition de projet <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($project_id), get_permalink($user_post_id), get_the_title($user_post_id) );
				break;
			case 'change_proposal_status':
				$project_id = !empty($args['project_id']) ? $args['project_id'] : '';
				$proposal_id = !empty($args['proposal_id']) ? $args['proposal_id'] : '';
				$user_id = !empty($args['user_id']) ? $args['user_id'] : '';
				
				$my_projects_page_id = wp_freeio_get_option('my_projects_page_id');
				$my_projects_url = get_permalink( $my_projects_page_id );
				$my_projects_url = add_query_arg( 'project_id', $project_id, remove_query_arg( 'project_id', $my_projects_url ) );
				$my_projects_url = add_query_arg( 'proposal_id', $proposal_id, remove_query_arg( 'proposal_id', $my_projects_url ) );
				$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_projects_url ) );
				
				$post_type = get_post_status($project_id);
				if ( $post_type == 'completed' ) {
					$html = sprintf(__('Votre projet est terminé : <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($project_id) );
				} elseif ( $post_type == 'cancelled' ) {
					$html = sprintf(__('Votre projet a été annulé : <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($project_id) );
				} else {
					$html = '';
				}
				break;

			case 'hired_service':
				$service_id = !empty($args['service_id']) ? $args['service_id'] : '';
				$employer_user_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_user_id);

				$html = sprintf(__('Vous avez été embauché pour le service suivant <a href="%s">%s</a> par le client <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($service_id), get_the_title($service_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'cancelled_hired_service':
				$service_id = !empty($args['service_id']) ? $args['service_id'] : '';
				$employer_user_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_user_id);
				
				$html = sprintf(__('L\'embauche a été annulée pour le service suivant <a href="%s">%s</a> par le client <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($service_id), get_the_title($service_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'service_message':
				$post_type = !empty($args['post_type']) ? $args['post_type'] : '';
				$service_id = !empty($args['service_id']) ? $args['service_id'] : '';
				$service_order_id = !empty($args['service_order_id']) ? $args['service_order_id'] : '';
				$user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				
				$view_history_url = '';
				if ( $post_type == 'freelancer' ) {
					$my_services_page_id = wp_freeio_get_option('my_services_page_id');
					
					$my_services_url = get_permalink( $my_services_page_id );

					$my_services_url = add_query_arg( 'service_id', $service_id, remove_query_arg( 'service_id', $my_services_url ) );
					$my_services_url = add_query_arg( 'service_order_id', $service_order_id, remove_query_arg( 'service_order_id', $my_services_url ) );
					$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_services_url ) );

					
					$user_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
					$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
				} elseif ( $post_type == 'employer' ) {
					$my_services_page_id = wp_freeio_get_option('my_bought_services_page_id');
					$my_services_url = get_permalink( $my_services_page_id );

					$my_services_url = add_query_arg( 'service_id', $service_id, remove_query_arg( 'service_id', $my_services_url ) );
					$my_services_url = add_query_arg( 'service_order_id', $service_order_id, remove_query_arg( 'service_order_id', $my_services_url ) );
					$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_services_url ) );

					$user_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
					$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
				}

				$html = sprintf(__('Un message a été envoyé concernant votre service <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($service_id), get_permalink($user_post_id), get_the_title($user_post_id) );
				break;
			case 'change_service_status':
				$service_id = !empty($args['service_id']) ? $args['service_id'] : '';
				$service_order_id = !empty($args['service_order_id']) ? $args['service_order_id'] : '';
				
				$my_services_page_id = wp_freeio_get_option('my_services_page_id');
				$my_services_url = get_permalink( $my_services_page_id );
				$my_services_url = add_query_arg( 'service_id', $service_id, remove_query_arg( 'service_id', $my_services_url ) );
				$my_services_url = add_query_arg( 'service_order_id', $service_order_id, remove_query_arg( 'service_order_id', $my_services_url ) );
				$view_history_url = add_query_arg( 'action', 'view-history', remove_query_arg( 'action', $my_services_url ) );
				
				$post_type = get_post_status($service_order_id);
				if ( $post_type == 'completed' ) {
					$html = sprintf(__('Votre service est terminé : <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($service_id) );
				} elseif ( $post_type == 'cancelled' ) {
					$html = sprintf(__('Votre service a été annulé : <a href="%s">%s</a>.', 'wp-freeio'), $view_history_url, get_the_title($service_id) );
				} else {
					$html = '';
				}
				break;
			case 'new_dispute':
				$dispute_id = !empty($args['dispute_id']) ? $args['dispute_id'] : '';
				$user_post_type = !empty($args['post_type']) ? $args['post_type'] : '';
				if ( $user_post_type == 'employer') {
					$user_post_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				} else {
					$user_post_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				}
				$post_id = !empty($args['post_id']) ? $args['post_id'] : '';
				$p_post_type = get_post_type($post_id);
				if ( !empty($post_type) && $post_type === 'service_order' ) {
					$p_post_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id', true);
				} else {
					$p_post_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id', true);
				}

				$my_disputes_page_id = wp_freeio_get_option('my_disputes_page_id');
				$my_disputes_url = get_permalink( $my_disputes_page_id );

				$my_disputes_url = add_query_arg( 'dispute_id', $dispute_id, remove_query_arg( 'dispute_id', $my_disputes_url ) );
				$message_url = add_query_arg( 'action', 'view-detail', remove_query_arg( 'action', $my_disputes_url ) );
				

				$html = sprintf(__('Un litige a été soumis concernant <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $message_url, get_the_title($p_post_id), get_permalink($user_post_id), get_the_title($user_post_id) );
				break;
			case 'new_dispute_message':
				$dispute_id = !empty($args['dispute_id']) ? $args['dispute_id'] : '';
				$user_post_type = !empty($args['post_type']) ? $args['post_type'] : '';
				if ( $user_post_type == 'employer') {
					$user_post_id = !empty($args['freelancer_user_id']) ? $args['freelancer_user_id'] : '';
				} else {
					$user_post_id = !empty($args['employer_user_id']) ? $args['employer_user_id'] : '';
				}
				$post_id = !empty($args['post_id']) ? $args['post_id'] : '';
				$p_post_type = get_post_type($post_id);
				if ( !empty($post_type) && $post_type === 'service_order' ) {
					$p_post_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id', true);
				} else {
					$p_post_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id', true);
				}

				$my_disputes_page_id = wp_freeio_get_option('my_disputes_page_id');
				$my_disputes_url = get_permalink( $my_disputes_page_id );

				$my_disputes_url = add_query_arg( 'dispute_id', $dispute_id, remove_query_arg( 'dispute_id', $my_disputes_url ) );
				$message_url = add_query_arg( 'action', 'view-detail', remove_query_arg( 'action', $my_disputes_url ) );

				$html = sprintf(__('Un nouveau message a été envoyé concernant le litige <a href="%s">%s</a> par <a href="%s">%s</a>.', 'wp-freeio'), $message_url, get_the_title($p_post_id), get_permalink($user_post_id), get_the_title($user_post_id) );
				break;
			case 'review_employer':
				$employer_id = !empty($args['employer_id']) ? $args['employer_id'] : '';
				$review_id = !empty($args['review_id']) ? $args['review_id'] : '';
				$rating = !empty($args['rating']) ? $args['rating'] : '';
				$user_id = !empty($args['user_id']) ? $args['user_id'] : '';
				$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);

				$html = sprintf(__('Un avis a été soumis pour vous par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($freelancer_id), get_the_title($freelancer_id) );
				break;
			case 'review_freelancer':
				$freelancer_id = !empty($args['freelancer_id']) ? $args['freelancer_id'] : '';
				$review_id = !empty($args['review_id']) ? $args['review_id'] : '';
				$rating = !empty($args['rating']) ? $args['rating'] : '';
				$user_id = !empty($args['user_id']) ? $args['user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($user_id);

				$html = sprintf(__('Un avis a été soumis pour vous par <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'review_service':
				$service_id = !empty($args['service_id']) ? $args['service_id'] : '';
				$review_id = !empty($args['review_id']) ? $args['review_id'] : '';
				$rating = !empty($args['rating']) ? $args['rating'] : '';
				$user_id = !empty($args['user_id']) ? $args['user_id'] : '';
				$employer_id = WP_Freeio_User::get_employer_by_user_id($user_id);

				$html = sprintf(__('Un avis a été soumis pour votre service <a href="%s">%s</a> par le client <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($service_id), get_the_title($service_id), get_permalink($employer_id), get_the_title($employer_id) );
				break;
			case 'withdraw_approved':
				$amount = !empty($args['amount']) ? $args['amount'] : '';
				$withdraw_id = !empty($args['withdraw_id']) ? $args['withdraw_id'] : '';
				$html = sprintf(__('Votre demande de retrait de %s a été approuvée.', 'wp-freeio'), WP_Freeio_Price::format_price($amount) );
				break;
			case 'withdraw_cancelled':
				$amount = !empty($args['amount']) ? $args['amount'] : '';
				$withdraw_id = !empty($args['withdraw_id']) ? $args['withdraw_id'] : '';
				$html = sprintf(__('Votre demande de retrait de %s a été refusée.', 'wp-freeio'), WP_Freeio_Price::format_price($amount) );
				break;
			case 'new_invoice':
				$invoice_id = !empty($args['invoice_id']) ? $args['invoice_id'] : '';
				$html = sprintf(__('Une nouvelle facture a été générée : <a href="%s">%s</a>.', 'wp-freeio'), get_permalink($invoice_id), get_the_title($invoice_id) );
				break;
			default:
				$html = '';
				break;
		}

		return apply_filters('wp-freeio-get-display-notify', $html, $args);
	}

	public static function notify_employer_apply($employer_id, $job_id, $application_id, $is_internal = false) {
		if ( empty($employer_id) || empty($job_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => $is_internal ? 'internal_apply' : 'email_apply',
            'job_id' => $job_id,
            'application_id' => $application_id,
		);
		if ( $is_internal ) {
			$freelancer_id = WP_Freeio_Applicant::get_post_meta($application_id, 'freelancer_id');
			$notify_args['freelancer_id'] = $freelancer_id;
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_remove_apply($freelancer_id, $job_id, $application_id) {
		if ( empty($freelancer_id) || empty($job_id) ) {
			return;
		}

		$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'remove_apply',
            'job_id' => $job_id,
            'application_id' => $application_id,
            'employer_id' => $employer_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_reject_apply($employer_id, $job_id, $application_id) {
		if ( empty($employer_id) || empty($job_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'reject_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_reject_apply($freelancer_id, $job_id, $application_id) {
		if ( empty($freelancer_id) || empty($job_id) ) {
			return;
		}

		$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'reject_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
            'employer_id' => $employer_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_undo_reject_apply($employer_id, $job_id, $application_id) {
		if ( empty($employer_id) || empty($job_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'undo_reject_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_undo_reject_apply($freelancer_id, $job_id, $application_id) {
		if ( empty($freelancer_id) || empty($job_id) ) {
			return;
		}

		$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'undo_reject_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
            'employer_id' => $employer_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_approve_apply($employer_id, $job_id, $application_id) {
		if ( empty($employer_id) || empty($job_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'approve_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_approve_apply($freelancer_id, $job_id, $application_id) {
		if ( empty($freelancer_id) || empty($job_id) ) {
			return;
		}

		$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'approve_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
            'employer_id' => $employer_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_undo_approve_apply($employer_id, $job_id, $application_id) {
		if ( empty($employer_id) || empty($job_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'undo_approve_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_undo_approve_apply($freelancer_id, $job_id, $application_id) {
		if ( empty($freelancer_id) || empty($job_id) ) {
			return;
		}

		$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'undo_approve_applied',
            'job_id' => $job_id,
            'application_id' => $application_id,
            'employer_id' => $employer_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_new_meeting($post_id, $user_id) {
		$post_type = get_post_type($post_id);
		if ( $post_type == 'job_applicant' ) {
			$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
			$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Applicant::get_post_meta($post_id, 'freelancer_id');
		} elseif( $post_type == 'project_proposal' ) {
			$project_id = WP_Freeio_Proposal::get_post_meta($post_id, 'project_id');
			$employer_id = WP_Freeio_Project::get_post_meta($project_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Proposal::get_post_meta($post_id, 'freelancer_user_id');
			$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_id);
		} elseif( $post_type == 'service_order' ) {
			$service_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'service_id');
			$freelancer_id = WP_Freeio_Service::get_post_meta($service_id, 'author' );
			$employer_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'employer_user_id');
			$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_id);
		} else {
			return;
		}

		$meeting = !empty($_POST['meeting']) ? $_POST['meeting'] : '';
		$meeting_id = !empty($meeting['ID']) ? $meeting['ID'] : '';
		if ( $user_id == $employer_id ) {
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'create_meeting',
	            'post_id' => $post_id,
	            'employer_id' => $employer_id,
			);
		} else {
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'create_meeting',
	            'post_id' => $post_id,
	            'freelancer_id' => $freelancer_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_reschedule_meeting($meeting_id, $user_id) {
		$post_id = WP_Freeio_Meeting::get_post_meta($meeting_id, 'post_id');
		$post_type = get_post_type($post_id);
		if ( $post_type == 'job_applicant' ) {
			$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
			$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Applicant::get_post_meta($post_id, 'freelancer_id');
		} elseif( $post_type == 'project_proposal' ) {
			$project_id = WP_Freeio_Proposal::get_post_meta($post_id, 'project_id');
			$employer_id = WP_Freeio_Project::get_post_meta($project_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Proposal::get_post_meta($post_id, 'freelancer_user_id');
			$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_id);
		} elseif( $post_type == 'service_order' ) {
			$service_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'service_id');
			$freelancer_id = WP_Freeio_Service::get_post_meta($service_id, 'author' );
			$employer_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'employer_user_id');
			$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_id);
		} else {
			return;
		}

		if ( $user_id == $employer_id ) {
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'reschedule_meeting',
	            'meeting_id' => $meeting_id,
	            'reschedule_user_id' => $user_id,
			);
		} else {
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'reschedule_meeting',
	            'meeting_id' => $meeting_id,
	            'reschedule_user_id' => $user_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_remove_meeting($post_id, $user_id) {
		$post_type = get_post_type($post_id);
		if ( $post_type == 'job_applicant' ) {
			$job_id = WP_Freeio_Applicant::get_post_meta($post_id, 'job_id');
			$employer_id = WP_Freeio_Job_Listing::get_post_meta($job_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Applicant::get_post_meta($post_id, 'freelancer_id');
		} elseif( $post_type == 'project_proposal' ) {
			$project_id = WP_Freeio_Proposal::get_post_meta($post_id, 'project_id');
			$employer_id = WP_Freeio_Project::get_post_meta($project_id, 'employer_post_id');
			$freelancer_id = WP_Freeio_Proposal::get_post_meta($post_id, 'freelancer_user_id');
			$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_id);
		} elseif( $post_type == 'service_order' ) {
			$service_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'service_id');
			$freelancer_id = WP_Freeio_Service::get_post_meta($service_id, 'author' );
			$employer_id = WP_Freeio_Service_Order::get_post_meta($post_id, 'employer_user_id');
			$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_id);
		} else {
			return;
		}

		if ( $user_id == $employer_id ) {
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'remove_meeting',
	            'post_id' => $post_id,
	            'employer_id' => $employer_id,
			);
		} else {
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'cancel_meeting',
	            'post_id' => $post_id,
	            'freelancer_id' => $freelancer_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_invite_apply($freelancer_id, $project_ids) {
		if ( empty($freelancer_id) || empty($project_ids) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'invite_freelancer_apply',
            'project_ids' => $project_ids,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_new_proposal($project_id, $proposal_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		$employer_id = WP_Freeio_Project::get_post_meta($project_id, 'employer_post_id');
		$freelancer_user_id = get_post_field('post_author', $proposal_id);
		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'new_proposal',
            'project_id' => $project_id,
            'proposal_id' => $proposal_id,
            'freelancer_user_id' => $freelancer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_employer_edit_proposal($project_id, $proposal_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		$employer_id = WP_Freeio_Project::get_post_meta($project_id, 'employer_post_id');
		$freelancer_user_id = get_post_field('post_author', $proposal_id);
		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'edit_proposal',
            'project_id' => $project_id,
            'proposal_id' => $proposal_id,
            'freelancer_user_id' => $freelancer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_hired_proposal($project_id, $proposal_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		$employer_user_id = get_post_field('post_author', $project_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'hired_proposal',
            'project_id' => $project_id,
            'proposal_id' => $proposal_id,
            'employer_user_id' => $employer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_cancelled_hired_proposal($project_id, $proposal_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		$employer_user_id = get_post_field('post_author', $project_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'cancelled_hired_proposal',
            'project_id' => $project_id,
            'proposal_id' => $proposal_id,
            'employer_user_id' => $employer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_proposal_message($post_type, $project_id, $proposal_id, $user_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		if ( $post_type == 'freelancer' ) {
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'proposal_message',
	            'project_id' => $project_id,
	            'proposal_id' => $proposal_id,
	            'freelancer_user_id' => $user_id,
			);
		} else {
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'proposal_message',
	            'project_id' => $project_id,
	            'proposal_id' => $proposal_id,
	            'user_post_id' => $user_post_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_change_proposal_status($project_id, $proposal_id) {
		if ( empty($project_id) || empty($proposal_id) ) {
			return;
		}
		$freelancer_user_id = get_post_field('post_author', $proposal_id);
		$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_user_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'change_proposal_status',
            'project_id' => $project_id,
            'proposal_id' => $proposal_id,
            'user_id' => $freelancer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_hired_service($service_id, $service_order_id) {
		if ( empty($service_id) || empty($service_order_id) ) {
			return;
		}
		$employer_user_id = get_post_field('post_author', $service_order_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'hired_service',
            'service_id' => $service_id,
            'service_order_id' => $service_order_id,
            'employer_user_id' => $employer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_freelancer_cancelled_hired_service($service_id, $service_order_id) {
		if ( empty($service_id) || empty($service_order_id) ) {
			return;
		}
		$employer_user_id = get_post_field('post_author', $service_order_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'cancelled_hired_service',
            'service_id' => $service_id,
            'service_order_id' => $service_order_id,
            'employer_user_id' => $employer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_service_message($post_type, $service_id, $service_order_id, $user_id) {
		if ( empty($service_id) || empty($service_order_id) ) {
			return;
		}
		if ( $post_type == 'freelancer' ) {
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'service_message',
	            'service_id' => $service_id,
	            'service_order_id' => $service_order_id,
	            'employer_user_id' => $user_id,
			);
		} else {
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'service_message',
	            'service_id' => $service_id,
	            'service_order_id' => $service_order_id,
	            'freelancer_user_id' => $user_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_change_service_status($service_id, $service_order_id) {
		if ( empty($service_id) || empty($service_order_id) ) {
			return;
		}
		$employer_user_id = get_post_field('post_author', $service_order_id);
		$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_user_id);
		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'change_service_status',
            'service_id' => $service_id,
            'service_order_id' => $service_order_id,
            'user_id' => $employer_user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_new_dispute($dispute_id, $post_id, $user_id) {
		if ( empty($dispute_id) || empty($post_id) ) {
			return;
		}
		$post_type = get_post_type($post_id);
		if ( $post_type == 'service_order' ) {
			$service_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id', true);
			$freelancer_id = get_post_field('post_author', $service_id);
			$employer_id = get_post_field('post_author', $post_id);
			$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_id);
		} else {
			$project_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id', true);
			$employer_id = get_post_field('post_author', $project_id);
			$freelancer_id = get_post_field('post_author', $post_id);
			$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_id);
		}

		if ( $user_id == $employer_id ) {
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'new_dispute',
	            'dispute_id' => $dispute_id,
	            'post_id' => $post_id,
	            'employer_user_id' => $user_id,
			);
		} else {
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'new_dispute',
	            'dispute_id' => $dispute_id,
	            'post_id' => $post_id,
	            'freelancer_user_id' => $user_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_new_dispute_message($dispute_id, $post_id, $user_id) {
		if ( empty($dispute_id) || empty($post_id) ) {
			return;
		}
		$post_type = get_post_type($post_id);
		if ( $post_type == 'service_order' ) {
			$service_id = get_post_meta($post_id, WP_FREEIO_SERVICE_ORDER_PREFIX.'service_id', true);
			$freelancer_id = get_post_field('post_author', $service_id);
			$employer_id = get_post_field('post_author', $post_id);
			$employer_id = WP_Freeio_User::get_employer_by_user_id($employer_id);
		} else {
			$project_id = get_post_meta($post_id, WP_FREEIO_PROJECT_PROPOSAL_PREFIX.'project_id', true);
			$employer_id = get_post_field('post_author', $project_id);
			$freelancer_id = get_post_field('post_author', $post_id);
			$freelancer_id = WP_Freeio_User::get_freelancer_by_user_id($freelancer_id);
		}

		if ( $user_id == $employer_id ) {
			$notify_args = array(
				'post_type' => 'freelancer',
				'user_post_id' => $freelancer_id,
	            'type' => 'new_dispute_message',
	            'dispute_id' => $dispute_id,
	            'post_id' => $post_id,
	            'employer_user_id' => $user_id,
			);
		} else {
			$notify_args = array(
				'post_type' => 'employer',
				'user_post_id' => $employer_id,
	            'type' => 'new_dispute_message',
	            'dispute_id' => $dispute_id,
	            'post_id' => $post_id,
	            'freelancer_user_id' => $user_id,
			);
		}
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_review_employer($employer_id, $review_id, $rating, $user_id) {
		if ( empty($employer_id) || empty($review_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'employer',
			'user_post_id' => $employer_id,
            'type' => 'review_employer',
            'review_id' => $review_id,
            'rating' => $rating,
            'user_id' => $user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_review_freelancer($freelancer_id, $review_id, $rating, $user_id) {
		if ( empty($freelancer_id) || empty($review_id) ) {
			return;
		}

		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'review_freelancer',
            'review_id' => $review_id,
            'rating' => $rating,
            'user_id' => $user_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_review_service($service_id, $review_id, $rating, $user_id) {
		if ( empty($service_id) || empty($review_id) ) {
			return;
		}

		$freelancer_id = get_post_field('post_author', $service_id);
		$notify_args = array(
			'post_type' => 'freelancer',
			'user_post_id' => $freelancer_id,
            'type' => 'review_service',
            'review_id' => $review_id,
            'rating' => $rating,
            'user_id' => $user_id,
            'service_id' => $service_id,
		);
		WP_Freeio_User_Notification::add_notification($notify_args);
	}

	public static function notify_withdraw_approved($user_id, $withdraw_id, $amount) {
		$user_post_id = 0;
		$post_type = '';
		if ( WP_Freeio_User::is_employer($user_id) ) {
			$post_type = 'employer';
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
		} elseif ( WP_Freeio_User::is_freelancer($user_id) ) {
			$post_type = 'freelancer';
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
		}
		if ( $user_post_id ) {
			$notify_args = array(
				'post_type' => $post_type,
				'user_post_id' => $user_post_id,
	            'type' => 'withdraw_approved',
	            'withdraw_id' => $withdraw_id,
	            'amount' => $amount,
			);
			WP_Freeio_User_Notification::add_notification($notify_args);
		}
	}

	public static function notify_withdraw_cancelled($user_id, $withdraw_id, $amount) {
		$user_post_id = 0;
		$post_type = '';
		if ( WP_Freeio_User::is_employer($user_id) ) {
			$post_type = 'employer';
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
		} elseif ( WP_Freeio_User::is_freelancer($user_id) ) {
			$post_type = 'freelancer';
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
		}
		if ( $user_post_id ) {
			$notify_args = array(
				'post_type' => $post_type,
				'user_post_id' => $user_post_id,
	            'type' => 'withdraw_cancelled',
	            'withdraw_id' => $withdraw_id,
	            'amount' => $amount,
			);
			WP_Freeio_User_Notification::add_notification($notify_args);
		}
	}

	public static function notify_new_invoice($user_id, $invoice_id) {
		$user_post_id = 0;
		$post_type = '';
		if ( WP_Freeio_User::is_employer($user_id) ) {
			$post_type = 'employer';
			$user_post_id = WP_Freeio_User::get_employer_by_user_id($user_id);
		} elseif ( WP_Freeio_User::is_freelancer($user_id) ) {
			$post_type = 'freelancer';
			$user_post_id = WP_Freeio_User::get_freelancer_by_user_id($user_id);
		}
		if ( $user_post_id ) {
			$notify_args = array(
				'post_type' => $post_type,
				'user_post_id' => $user_post_id,
	            'type' => 'new_invoice',
	            'invoice_id' => $invoice_id,
			);
			WP_Freeio_User_Notification::add_notification($notify_args);
		}
	}

}

WP_Freeio_User_Notification::init();